/* main.c - Hello World + PLL + Interrupt example for S32R274 */
/* Description:  Drive CLKOUT0 to external pin with multiple cores */
/* Rev 1.0 Aug 10 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2016 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Aug-2-2018
* Classification    General Business Information
* Brief             S32R274 cores produces different CLKOUTs and takes turns
* 					controlling the pin
********************************************************************************
* Detailed Description:
* S32R274's default clock source is 16 MHz IRCOSC.  This code example
* configures the PLL0 to 240 MHz, PLL1 to 160 MHz, and SDPLL to zzz MHz.
* The system clock selector selects PLL0_PHI_CLK as its source. The clock
* domain SYS_CLK divides that by 3, such that SYS_CLK = 60 MHz.
* Each of S32R's three cores configures a different clock source
* for the CLKOUT0 pin. The z4 core drives PLL0_PHI_CLK out divided by 20
* (240 MHz/20 = 12 MHz), the z7a core drives PLL1_PHI_CLK divided by 20
* (160 MHz/20 = 8 MHz), and z7b drives XOSC divided by 20 (20 MHz/20 = 1MHz).
* The cores take turns controlling the CLKOUT0 pin my means of a
* PIT interrupt. The PIT is configured to interrupt every 5 seconds.
* The PIT interrupt fires in a round robin fashion amongst the cores
* such that each core's PIT interrupt routine is executed one after
* another, wherein the CLKOUT0 clock source changes. The switch is
* indicated in the oscilloscope as well as the user LED.
*
* Each core will configure a different GPIO to toggle.  When these GPIO
* pins are connected to user LEDs, you can observe a toggle.  The
* frequency of the toggle is based on an internal counter.
*
* Refer to "Z4 Waveform.png", "Z7a Waveform.png", and "Z7b Waveform.png"
* for the expected output of each core.
*
* NOTE: Make sure to start running the Z7a and Z7b threads before
* starting the Z4 thread.
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB + MPC57XXMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            240 MHz PLL
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Connect pin P9.7 (CLK_OUT0) to an oscilloscope to check
* 					CLK_OUT0 frequency. Connect P8.1 (PA0) to LED P7.1 (D2),
* 					P8.2 (PA1) to P7.2 (D3), and P8.3 (PA2) to P7.3 (D4).
*
* 					Refer to "Hello_World_PLL_Interrupt_Hardware_Setup.jpg"
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-10-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */

#define COUNTER_MAX 500000

extern void xcptn_xmpl(void);

void CLKOUT0_Init_Z7a(void);

void PIT_InterruptHandler(void);

void CLKOUT0_Init_Z7a(){

	/* MCB controls CLKOUT source selects. */
	MCB.CLKOUT_SEL.B.CLKOUT0 = 0b0100; //Select PLL1 for CLKOUT0
	CGM.AC14_DC0.R = 0x80130000; //Divide by (19+1) = 20 for output of 8 MHz

	/* Enable GPIO PA1 (P8.2) to flash */
	SIUL2.MSCR[1].B.OBE = 1;
	SIUL2.GPDO[0].B.PDO_4n1 = 1;

	/* Disable all other GPIOs */
	SIUL2.GPDO[0].B.PDO_4n = 1;
	SIUL2.MSCR[0].B.OBE = 0;

	SIUL2.GPDO[0].B.PDO_4n2 = 1;
	SIUL2.MSCR[2].B.OBE = 0;
}

void PIT_InterruptHandler(){
	/* Acknowledge the flag */
	PIT_0.TIMER[0].TFLG.R = 0x00000001; //w1c

	/* Configure CLKOUT0 so it displays PLL1_PHI_CLK,
	 * divided by 20 (8 MHz output).
	 */
	CLKOUT0_Init_Z7a();

	/* Change the interrupt destination so that
	 * the next time the interrupt fires, it
	 * goes to the z7b core
	 */
	INTC.PSR[226].R = 0x200F;  //Configure PIT_0 Ch0 interrupt in INTC. Set priority 15 and send to Core 2
}

int main(void)
{
	volatile int counter = 0;
	
	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	/* Loop forever */
	for(;;) {	   
	   	counter++;

	   	/* Wait 500000 counts of internal counter
	   	 * before toggling GPIO.
	   	 */
	   	if(counter >= COUNTER_MAX){
	   		SIUL2.GPDO[0].B.PDO_4n1 ^= 1; //Toggle PA1
	   		counter = 0; //Reset counter
	   	}
	}
}
